/*
 * The MIT License (MIT)
 *
 * Copyright © 2015-2017, Heiko Brumme
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package slick;

import java.nio.FloatBuffer;

/**
 * This class represents a 4x4-Matrix. GLSL equivalent to mat4.
 *
 * @author Heiko Brumme
 */
public class Matrix4f {

    private float m00, m01, m02, m03;
    private float m10, m11, m12, m13;
    private float m20, m21, m22, m23;
    private float m30, m31, m32, m33;
    
    byte PROPERTY_PERSPECTIVE = 1<<0;
    byte PROPERTY_IDENTITY = 1<<2;
    byte PROPERTY_AFFINE = 1<<1;
    byte PROPERTY_TRANSLATION = 1<<3;
    byte PROPERTY_ORTHONORMAL = 1<<4;
    
    int properties = 0;

    /**
     * Creates a 4x4 identity matrix.
     */
    public Matrix4f() {
        setIdentity();
    }

    /**
     * Creates a 4x4 matrix with specified columns.
     *
     * @param col1 Vector with values of the first column
     * @param col2 Vector with values of the second column
     * @param col3 Vector with values of the third column
     * @param col4 Vector with values of the fourth column
     */
    public Matrix4f(Vector4f col1, Vector4f col2, Vector4f col3, Vector4f col4) {
        m00 = col1.x;
        m10 = col1.y;
        m20 = col1.z;
        m30 = col1.w;

        m01 = col2.x;
        m11 = col2.y;
        m21 = col2.z;
        m31 = col2.w;

        m02 = col3.x;
        m12 = col3.y;
        m22 = col3.z;
        m32 = col3.w;

        m03 = col4.x;
        m13 = col4.y;
        m23 = col4.z;
        m33 = col4.w;
    }

    /**
     * Sets this matrix to the identity matrix.
     */
    public final void setIdentity() {
        m00 = 1f;
        m11 = 1f;
        m22 = 1f;
        m33 = 1f;

        m01 = 0f;
        m02 = 0f;
        m03 = 0f;
        m10 = 0f;
        m12 = 0f;
        m13 = 0f;
        m20 = 0f;
        m21 = 0f;
        m23 = 0f;
        m30 = 0f;
        m31 = 0f;
        m32 = 0f;
    }

    /**
     * Adds this matrix to another matrix.
     *
     * @param other The other matrix
     *
     * @return Sum of this + other
     */
    public Matrix4f add(Matrix4f other) {
        Matrix4f result = new Matrix4f();

        result.m00 = this.m00 + other.m00;
        result.m10 = this.m10 + other.m10;
        result.m20 = this.m20 + other.m20;
        result.m30 = this.m30 + other.m30;

        result.m01 = this.m01 + other.m01;
        result.m11 = this.m11 + other.m11;
        result.m21 = this.m21 + other.m21;
        result.m31 = this.m31 + other.m31;

        result.m02 = this.m02 + other.m02;
        result.m12 = this.m12 + other.m12;
        result.m22 = this.m22 + other.m22;
        result.m32 = this.m32 + other.m32;

        result.m03 = this.m03 + other.m03;
        result.m13 = this.m13 + other.m13;
        result.m23 = this.m23 + other.m23;
        result.m33 = this.m33 + other.m33;

        return result;
    }

    /**
     * Negates this matrix.
     *
     * @return Negated matrix
     */
    public Matrix4f negate() {
        return multiply(-1f);
    }

    /**
     * Subtracts this matrix from another matrix.
     *
     * @param other The other matrix
     *
     * @return Difference of this - other
     */
    public Matrix4f subtract(Matrix4f other) {
        return this.add(other.negate());
    }

    /**
     * Multiplies this matrix with a scalar.
     *
     * @param scalar The scalar
     *
     * @return Scalar product of this * scalar
     */
    public Matrix4f multiply(float scalar) {
        Matrix4f result = new Matrix4f();

        result.m00 = this.m00 * scalar;
        result.m10 = this.m10 * scalar;
        result.m20 = this.m20 * scalar;
        result.m30 = this.m30 * scalar;

        result.m01 = this.m01 * scalar;
        result.m11 = this.m11 * scalar;
        result.m21 = this.m21 * scalar;
        result.m31 = this.m31 * scalar;

        result.m02 = this.m02 * scalar;
        result.m12 = this.m12 * scalar;
        result.m22 = this.m22 * scalar;
        result.m32 = this.m32 * scalar;

        result.m03 = this.m03 * scalar;
        result.m13 = this.m13 * scalar;
        result.m23 = this.m23 * scalar;
        result.m33 = this.m33 * scalar;

        return result;
    }

    /**
     * Multiplies this matrix to a vector.
     *
     * @param vector The vector
     *
     * @return Vector product of this * other
     */
    public Vector4f multiply(Vector4f vector) {
        float x = this.m00 * vector.x + this.m01 * vector.y + this.m02 * vector.z + this.m03 * vector.w;
        float y = this.m10 * vector.x + this.m11 * vector.y + this.m12 * vector.z + this.m13 * vector.w;
        float z = this.m20 * vector.x + this.m21 * vector.y + this.m22 * vector.z + this.m23 * vector.w;
        float w = this.m30 * vector.x + this.m31 * vector.y + this.m32 * vector.z + this.m33 * vector.w;
        return new Vector4f(x, y, z, w);
    }

    /**
     * Multiplies this matrix to another matrix.
     *
     * @param other The other matrix
     *
     * @return Matrix product of this * other
     */
    public Matrix4f multiply(Matrix4f other) {
        Matrix4f result = new Matrix4f();

        result.m00 = this.m00 * other.m00 + this.m01 * other.m10 + this.m02 * other.m20 + this.m03 * other.m30;
        result.m10 = this.m10 * other.m00 + this.m11 * other.m10 + this.m12 * other.m20 + this.m13 * other.m30;
        result.m20 = this.m20 * other.m00 + this.m21 * other.m10 + this.m22 * other.m20 + this.m23 * other.m30;
        result.m30 = this.m30 * other.m00 + this.m31 * other.m10 + this.m32 * other.m20 + this.m33 * other.m30;

        result.m01 = this.m00 * other.m01 + this.m01 * other.m11 + this.m02 * other.m21 + this.m03 * other.m31;
        result.m11 = this.m10 * other.m01 + this.m11 * other.m11 + this.m12 * other.m21 + this.m13 * other.m31;
        result.m21 = this.m20 * other.m01 + this.m21 * other.m11 + this.m22 * other.m21 + this.m23 * other.m31;
        result.m31 = this.m30 * other.m01 + this.m31 * other.m11 + this.m32 * other.m21 + this.m33 * other.m31;

        result.m02 = this.m00 * other.m02 + this.m01 * other.m12 + this.m02 * other.m22 + this.m03 * other.m32;
        result.m12 = this.m10 * other.m02 + this.m11 * other.m12 + this.m12 * other.m22 + this.m13 * other.m32;
        result.m22 = this.m20 * other.m02 + this.m21 * other.m12 + this.m22 * other.m22 + this.m23 * other.m32;
        result.m32 = this.m30 * other.m02 + this.m31 * other.m12 + this.m32 * other.m22 + this.m33 * other.m32;

        result.m03 = this.m00 * other.m03 + this.m01 * other.m13 + this.m02 * other.m23 + this.m03 * other.m33;
        result.m13 = this.m10 * other.m03 + this.m11 * other.m13 + this.m12 * other.m23 + this.m13 * other.m33;
        result.m23 = this.m20 * other.m03 + this.m21 * other.m13 + this.m22 * other.m23 + this.m23 * other.m33;
        result.m33 = this.m30 * other.m03 + this.m31 * other.m13 + this.m32 * other.m23 + this.m33 * other.m33;

        return result;
    }

    /**
     * Transposes this matrix.
     *
     * @return Transposed matrix
     */
    public Matrix4f transpose() {
        Matrix4f result = new Matrix4f();

        result.m00 = this.m00;
        result.m10 = this.m01;
        result.m20 = this.m02;
        result.m30 = this.m03;

        result.m01 = this.m10;
        result.m11 = this.m11;
        result.m21 = this.m12;
        result.m31 = this.m13;

        result.m02 = this.m20;
        result.m12 = this.m21;
        result.m22 = this.m22;
        result.m32 = this.m23;

        result.m03 = this.m30;
        result.m13 = this.m31;
        result.m23 = this.m32;
        result.m33 = this.m33;

        return result;
    }

    /**
     * Stores the matrix in a given Buffer.
     *
     * @param buffer The buffer to store the matrix data
     */
    public void toBuffer(FloatBuffer buffer) {
        buffer.put(m00).put(m10).put(m20).put(m30);
        buffer.put(m01).put(m11).put(m21).put(m31);
        buffer.put(m02).put(m12).put(m22).put(m32);
        buffer.put(m03).put(m13).put(m23).put(m33);
        buffer.flip();
    }

    /**
     * Creates a orthographic projection matrix. Similar to
     * <code>glOrtho(left, right, bottom, top, near, far)</code>.
     *
     * @param left   Coordinate for the left vertical clipping pane
     * @param right  Coordinate for the right vertical clipping pane
     * @param bottom Coordinate for the bottom horizontal clipping pane
     * @param top    Coordinate for the bottom horizontal clipping pane
     * @param near   Coordinate for the near depth clipping pane
     * @param far    Coordinate for the far depth clipping pane
     *
     * @return Orthographic matrix
     */
    public static Matrix4f orthographic(float left, float right, float bottom, float top, float near, float far) {
        Matrix4f ortho = new Matrix4f();

        float tx = -(right + left) / (right - left);
        float ty = -(top + bottom) / (top - bottom);
        float tz = -(far + near) / (far - near);

        ortho.m00 = 2f / (right - left);
        ortho.m11 = 2f / (top - bottom);
        ortho.m22 = -2f / (far - near);
        ortho.m03 = tx;
        ortho.m13 = ty;
        ortho.m23 = tz;

        return ortho;
    }

    /**
     * Creates a perspective projection matrix. Similar to
     * <code>glFrustum(left, right, bottom, top, near, far)</code>.
     *
     * @param left   Coordinate for the left vertical clipping pane
     * @param right  Coordinate for the right vertical clipping pane
     * @param bottom Coordinate for the bottom horizontal clipping pane
     * @param top    Coordinate for the bottom horizontal clipping pane
     * @param near   Coordinate for the near depth clipping pane, must be
     *               positive
     * @param far    Coordinate for the far depth clipping pane, must be
     *               positive
     *
     * @return Perspective matrix
     */
    public static Matrix4f frustum(float left, float right, float bottom, float top, float near, float far) {
        Matrix4f frustum = new Matrix4f();

        float a = (right + left) / (right - left);
        float b = (top + bottom) / (top - bottom);
        float c = -(far + near) / (far - near);
        float d = -(2f * far * near) / (far - near);

        frustum.m00 = (2f * near) / (right - left);
        frustum.m11 = (2f * near) / (top - bottom);
        frustum.m02 = a;
        frustum.m12 = b;
        frustum.m22 = c;
        frustum.m32 = -1f;
        frustum.m23 = d;
        frustum.m33 = 0f;

        return frustum;
    }

    /**
     * Creates a perspective projection matrix. Similar to
     * <code>gluPerspective(fovy, aspec, zNear, zFar)</code>.
     *
     * @param fovy   Field of view angle in degrees
     * @param aspect The aspect ratio is the ratio of width to height
     * @param near   Distance from the viewer to the near clipping plane, must
     *               be positive
     * @param far    Distance from the viewer to the far clipping plane, must be
     *               positive
     *
     * @return Perspective matrix
     */
    public static Matrix4f perspective(float fovy, float aspect, float near, float far) {
        Matrix4f perspective = new Matrix4f();

        float f = (float) (1f / Math.tan(Math.toRadians(fovy) / 2f));

        perspective.m00 = f / aspect;
        perspective.m11 = f;
        perspective.m22 = (far + near) / (near - far);
        perspective.m32 = -1f;
        perspective.m23 = (2f * far * near) / (near - far);
        perspective.m33 = 0f;

        return perspective;
    }

    /**
     * Creates a translation matrix. Similar to
     * <code>glTranslate(x, y, z)</code>.
     *
     * @param x x coordinate of translation vector
     * @param y y coordinate of translation vector
     * @param z z coordinate of translation vector
     *
     * @return Translation matrix
     */
    public Matrix4f translate(float x, float y, float z) {
        Matrix4f translation = new Matrix4f();

        translation.m03 = x;
        translation.m13 = y;
        translation.m23 = z;

        return translation;
    }

    /**
     * Creates a rotation matrix. Similar to
     * <code>glRotate(angle, x, y, z)</code>.
     *
     * @param angle Angle of rotation in degrees
     * @param x     x coordinate of the rotation vector
     * @param y     y coordinate of the rotation vector
     * @param z     z coordinate of the rotation vector
     *
     * @return Rotation matrix
     */
    public static Matrix4f rotate(float angle, float x, float y, float z) {
        Matrix4f rotation = new Matrix4f();

        float c = (float) Math.cos(Math.toRadians(angle));
        float s = (float) Math.sin(Math.toRadians(angle));
        Vector3f vec = new Vector3f(x, y, z);
        if (vec.length() != 1f) {
            vec = vec.normalize();
            x = vec.x;
            y = vec.y;
            z = vec.z;
        }

        rotation.m00 = x * x * (1f - c) + c;
        rotation.m10 = y * x * (1f - c) + z * s;
        rotation.m20 = x * z * (1f - c) - y * s;
        rotation.m01 = x * y * (1f - c) - z * s;
        rotation.m11 = y * y * (1f - c) + c;
        rotation.m21 = y * z * (1f - c) + x * s;
        rotation.m02 = x * z * (1f - c) + y * s;
        rotation.m12 = y * z * (1f - c) - x * s;
        rotation.m22 = z * z * (1f - c) + c;

        return rotation;
    }

    /**
     * Creates a scaling matrix. Similar to <code>glScale(x, y, z)</code>.
     *
     * @param x Scale factor along the x coordinate
     * @param y Scale factor along the y coordinate
     * @param z Scale factor along the z coordinate
     *
     * @return Scaling matrix
     */
    public static Matrix4f scale(float x, float y, float z) {
        Matrix4f scaling = new Matrix4f();

        scaling.m00 = x;
        scaling.m11 = y;
        scaling.m22 = z;

        return scaling;
    }
    
    public FloatBuffer get(FloatBuffer buffer) {
        put(this, buffer.position(), buffer);
        return buffer;
    }

    public void put(Matrix4f m, int offset, FloatBuffer dest) {
        if (offset == 0)
            put0(m, dest);
        else
            putN(m, offset, dest);
    }
    public void put0(Matrix4f m, FloatBuffer dest) {
        dest.put(0,  m.m00())
        .put(1,  m.m01())
        .put(2,  m.m02())
        .put(3,  m.m03())
        .put(4,  m.m10())
        .put(5,  m.m11())
        .put(6,  m.m12())
        .put(7,  m.m13())
        .put(8,  m.m20())
        .put(9,  m.m21())
        .put(10, m.m22())
        .put(11, m.m23())
        .put(12, m.m30())
        .put(13, m.m31())
        .put(14, m.m32())
        .put(15, m.m33());
    }

    public void putN(Matrix4f m, int offset, FloatBuffer dest) {
        dest.put(offset,    m.m00())
        .put(offset+1,  m.m01())
        .put(offset+2,  m.m02())
        .put(offset+3,  m.m03())
        .put(offset+4,  m.m10())
        .put(offset+5,  m.m11())
        .put(offset+6,  m.m12())
        .put(offset+7,  m.m13())
        .put(offset+8,  m.m20())
        .put(offset+9,  m.m21())
        .put(offset+10, m.m22())
        .put(offset+11, m.m23())
        .put(offset+12, m.m30())
        .put(offset+13, m.m31())
        .put(offset+14, m.m32())
        .put(offset+15, m.m33());
    }

    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#properties()
     */
    public int properties() {
        return properties;
    }

    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m00()
     */
    public float m00() {
        return m00;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m01()
     */
    public float m01() {
        return m01;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m02()
     */
    public float m02() {
        return m02;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m03()
     */
    public float m03() {
        return m03;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m10()
     */
    public float m10() {
        return m10;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m11()
     */
    public float m11() {
        return m11;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m12()
     */
    public float m12() {
        return m12;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m13()
     */
    public float m13() {
        return m13;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m20()
     */
    public float m20() {
        return m20;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m21()
     */
    public float m21() {
        return m21;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m22()
     */
    public float m22() {
        return m22;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m23()
     */
    public float m23() {
        return m23;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m30()
     */
    public float m30() {
        return m30;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m31()
     */
    public float m31() {
        return m31;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m32()
     */
    public float m32() {
        return m32;
    }
    /* (non-Javadoc)
     * @see org.joml.Matrix4fc#m33()
     */
    public float m33() {
        return m33;
    }
    
    public void zero(Matrix4f dest) {
        dest._m00(0.0f)
        ._m01(0.0f)
        ._m02(0.0f)
        ._m03(0.0f)
        ._m10(0.0f)
        ._m11(0.0f)
        ._m12(0.0f)
        ._m13(0.0f)
        ._m20(0.0f)
        ._m21(0.0f)
        ._m22(0.0f)
        ._m23(0.0f)
        ._m30(0.0f)
        ._m31(0.0f)
        ._m32(0.0f)
        ._m33(0.0f);
    }

    /**
     * Set the value of the matrix element at column 0 and row 0 without updating the properties of the matrix.
     * 
     * @param m00
     *          the new value
     * @return this
     */
    Matrix4f _m00(float m00) {
        this.m00 = m00;
        return this;
    }
    /**
     * Set the value of the matrix element at column 0 and row 1 without updating the properties of the matrix.
     * 
     * @param m01
     *          the new value
     * @return this
     */
    Matrix4f _m01(float m01) {
        this.m01 = m01;
        return this;
    }
    /**
     * Set the value of the matrix element at column 0 and row 2 without updating the properties of the matrix.
     * 
     * @param m02
     *          the new value
     * @return this
     */
    Matrix4f _m02(float m02) {
        this.m02 = m02;
        return this;
    }
    /**
     * Set the value of the matrix element at column 0 and row 3 without updating the properties of the matrix.
     * 
     * @param m03
     *          the new value
     * @return this
     */
    Matrix4f _m03(float m03) {
        this.m03 = m03;
        return this;
    }
    /**
     * Set the value of the matrix element at column 1 and row 0 without updating the properties of the matrix.
     * 
     * @param m10
     *          the new value
     * @return this
     */
    Matrix4f _m10(float m10) {
        this.m10 = m10;
        return this;
    }
    /**
     * Set the value of the matrix element at column 1 and row 1 without updating the properties of the matrix.
     * 
     * @param m11
     *          the new value
     * @return this
     */
    Matrix4f _m11(float m11) {
        this.m11 = m11;
        return this;
    }
    /**
     * Set the value of the matrix element at column 1 and row 2 without updating the properties of the matrix.
     * 
     * @param m12
     *          the new value
     * @return this
     */
    Matrix4f _m12(float m12) {
        this.m12 = m12;
        return this;
    }
    /**
     * Set the value of the matrix element at column 1 and row 3 without updating the properties of the matrix.
     * 
     * @param m13
     *          the new value
     * @return this
     */
    Matrix4f _m13(float m13) {
        this.m13 = m13;
        return this;
    }
    /**
     * Set the value of the matrix element at column 2 and row 0 without updating the properties of the matrix.
     * 
     * @param m20
     *          the new value
     * @return this
     */
    Matrix4f _m20(float m20) {
        this.m20 = m20;
        return this;
    }
    /**
     * Set the value of the matrix element at column 2 and row 1 without updating the properties of the matrix.
     * 
     * @param m21
     *          the new value
     * @return this
     */
    Matrix4f _m21(float m21) {
        this.m21 = m21;
        return this;
    }
    /**
     * Set the value of the matrix element at column 2 and row 2 without updating the properties of the matrix.
     * 
     * @param m22
     *          the new value
     * @return this
     */
    Matrix4f _m22(float m22) {
        this.m22 = m22;
        return this;
    }
    /**
     * Set the value of the matrix element at column 2 and row 3 without updating the properties of the matrix.
     * 
     * @param m23
     *          the new value
     * @return this
     */
    Matrix4f _m23(float m23) {
        this.m23 = m23;
        return this;
    }
    /**
     * Set the value of the matrix element at column 3 and row 0 without updating the properties of the matrix.
     * 
     * @param m30
     *          the new value
     * @return this
     */
    Matrix4f _m30(float m30) {
        this.m30 = m30;
        return this;
    }
    /**
     * Set the value of the matrix element at column 3 and row 1 without updating the properties of the matrix.
     * 
     * @param m31
     *          the new value
     * @return this
     */
    Matrix4f _m31(float m31) {
        this.m31 = m31;
        return this;
    }
    /**
     * Set the value of the matrix element at column 3 and row 2 without updating the properties of the matrix.
     * 
     * @param m32
     *          the new value
     * @return this
     */
    Matrix4f _m32(float m32) {
        this.m32 = m32;
        return this;
    }

    /**
     * Set the value of the matrix element at column 3 and row 3 without updating the properties of the matrix.
     * 
     * @param m33
     *          the new value
     * @return this
     */
    Matrix4f _m33(float m33) {
        this.m33 = m33;
        return this;
    }
    
    Matrix4f _properties(int properties) {
        this.properties = properties;
        return this;
    }
    
    public Matrix4f identity() {
        if ((properties & PROPERTY_IDENTITY) != 0)
            return this;
        return
        _m00(1.0f).
        _m01(0.0f).
        _m02(0.0f).
        _m03(0.0f).
        _m10(0.0f).
        _m11(1.0f).
        _m12(0.0f).
        _m13(0.0f).
        _m20(0.0f).
        _m21(0.0f).
        _m22(1.0f).
        _m23(0.0f).
        _m30(0.0f).
        _m31(0.0f).
        _m32(0.0f).
        _m33(1.0f).
        _properties(PROPERTY_IDENTITY | PROPERTY_AFFINE | PROPERTY_TRANSLATION | PROPERTY_ORTHONORMAL);
    }


    public Matrix4f setPerspective(float fovy, float aspect, float zNear, float zFar, boolean zZeroToOne) {
        zero(this);
        float h = (float) Math.tan(fovy * 0.5f);
        this._m00(1.0f / (h * aspect))
            ._m11(1.0f / h);
        boolean farInf = zFar > 0 && Float.isInfinite(zFar);
        boolean nearInf = zNear > 0 && Float.isInfinite(zNear);
        if (farInf) {
            // See: "Infinite Projection Matrix" (http://www.terathon.com/gdc07_lengyel.pdf)
            float e = 1E-6f;
            this._m22(e - 1.0f)
                ._m32((e - (zZeroToOne ? 1.0f : 2.0f)) * zNear);
        } else if (nearInf) {
            float e = 1E-6f;
            this._m22((zZeroToOne ? 0.0f : 1.0f) - e)
                ._m32(((zZeroToOne ? 1.0f : 2.0f) - e) * zFar);
        } else {
            this._m22((zZeroToOne ? zFar : zFar + zNear) / (zNear - zFar))
                ._m32((zZeroToOne ? zFar : zFar + zFar) * zNear / (zNear - zFar));
        }
        return this
        ._m23(-1.0f)
        ._properties(PROPERTY_PERSPECTIVE);
    }
    
    public void identity(Matrix4f dest) {
        dest._m00(1.0f)
        ._m01(0.0f)
        ._m02(0.0f)
        ._m03(0.0f)
        ._m10(0.0f)
        ._m11(1.0f)
        ._m12(0.0f)
        ._m13(0.0f)
        ._m20(0.0f)
        ._m21(0.0f)
        ._m22(1.0f)
        ._m23(0.0f)
        ._m30(0.0f)
        ._m31(0.0f)
        ._m32(0.0f)
        ._m33(1.0f);
    }



    public Matrix4f rotateX(float ang) {
        return rotateX(ang, this);
    }
    
    public Matrix4f rotationX(float ang) {
        float sin = (float) Math.sin(ang), cos = (float) Math.cos(ang);
        if ((properties & PROPERTY_IDENTITY) == 0)
            identity(this);
        this._m11(cos)._m12(sin)._m21(-sin)._m22(cos)._properties(PROPERTY_AFFINE | PROPERTY_ORTHONORMAL);
        return this;
    }

    public Matrix4f setTranslation(float x, float y, float z) {
        return this._m30(x)._m31(y)._m32(z)._properties(properties & ~(PROPERTY_PERSPECTIVE | PROPERTY_IDENTITY));
    }

    public Matrix4f rotateX(float ang, Matrix4f dest) {
        if ((properties & PROPERTY_IDENTITY) != 0)
            return dest.rotationX(ang);
        else if ((properties & PROPERTY_TRANSLATION) != 0) {
            float x = m30, y = m31, z = m32;
            return dest.rotationX(ang).setTranslation(x, y, z);
        }
        return rotateXInternal(ang, dest);
    }
    private Matrix4f rotateXInternal(float ang, Matrix4f dest) {
        float sin = (float) Math.sin(ang), cos = (float) Math.cos(ang);
        float lm10 = m10, lm11 = m11, lm12 = m12, lm13 = m13, lm20 = m20, lm21 = m21, lm22 = m22, lm23 = m23;
        return dest
        ._m20(lm10 * -sin + lm20 * cos)
        ._m21(lm11 * -sin + lm21 * cos)
        ._m22(lm12 * -sin + lm22 * cos)
        ._m23(lm13 * -sin + lm23 * cos)
        ._m10(lm10 * cos + lm20 * sin)
        ._m11(lm11 * cos + lm21 * sin)
        ._m12(lm12 * cos + lm22 * sin)
        ._m13(lm13 * cos + lm23 * sin)
        ._m00(m00)
        ._m01(m01)
        ._m02(m02)
        ._m03(m03)
        ._m30(m30)
        ._m31(m31)
        ._m32(m32)
        ._m33(m33)
        ._properties(properties & ~(PROPERTY_PERSPECTIVE | PROPERTY_IDENTITY | PROPERTY_TRANSLATION));
    }
    
    public Matrix4f rotateY(float ang) {
        return rotateY(ang, this);
    }

    public Matrix4f rotateY(float ang, Matrix4f dest) {
        if ((properties & PROPERTY_IDENTITY) != 0)
            return dest.rotationY(ang);
        else if ((properties & PROPERTY_TRANSLATION) != 0) {
            float x = m30, y = m31, z = m32;
            return dest.rotationY(ang).setTranslation(x, y, z);
        }
        return rotateYInternal(ang, dest);
    }
    private Matrix4f rotateYInternal(float ang, Matrix4f dest) {
        float sin = (float) Math.sin(ang);
        float cos = (float) Math.cos(ang);
        // add temporaries for dependent values
        float nm00 = m00 * cos + m20 * -sin;
        float nm01 = m01 * cos + m21 * -sin;
        float nm02 = m02 * cos + m22 * -sin;
        float nm03 = m03 * cos + m23 * -sin;
        // set non-dependent values directly
        return dest
        ._m20(m00 * sin + m20 * cos)
        ._m21(m01 * sin + m21 * cos)
        ._m22(m02 * sin + m22 * cos)
        ._m23(m03 * sin + m23 * cos)
        // set other values
        ._m00(nm00)
        ._m01(nm01)
        ._m02(nm02)
        ._m03(nm03)
        ._m10(m10)
        ._m11(m11)
        ._m12(m12)
        ._m13(m13)
        ._m30(m30)
        ._m31(m31)
        ._m32(m32)
        ._m33(m33)
        ._properties(properties & ~(PROPERTY_PERSPECTIVE | PROPERTY_IDENTITY | PROPERTY_TRANSLATION));
    }

    public Matrix4f rotationY(float ang) {
        float sin = (float) Math.sin(ang), cos = (float) Math.cos(ang);
        if ((properties & PROPERTY_IDENTITY) == 0)
            identity(this);
        this._m00(cos)._m02(-sin)._m20(sin)._m22(cos)._properties(PROPERTY_AFFINE | PROPERTY_ORTHONORMAL);
        return this;
    }

    public Vector3f positiveZ(Vector3f dir) {
        return dir.set(m10 * m21 - m11 * m20, m20 * m01 - m21 * m00, m00 * m11 - m01 * m10).normalize();
    }

    public Vector3f positiveX(Vector3f dir) {
        return dir.set(m11 * m22 - m12 * m21, m02 * m21 - m01 * m22, m01 * m12 - m02 * m11).normalize();
    }



}
